import type { Metadata } from 'next'
import { Inter, Montserrat, Cinzel, Orbitron, Noto_Sans_Georgian } from 'next/font/google'
import { NextIntlClientProvider } from 'next-intl'
import { getMessages, getTranslations } from 'next-intl/server'
import { notFound } from 'next/navigation'
import '../globals.css'
import { Navbar } from '@/components/Navbar'
import { Footer } from '@/components/Footer'
import { ScrollToTop } from '@/components/ScrollToTop'
import { Providers } from '@/components/Providers'
import { ScrollProgress } from '@/components/ScrollProgress'
import { CustomCursor } from '@/components/CustomCursor'
import { EasterEgg } from '@/components/EasterEgg'
import { FloatingCTA } from '@/components/FloatingCTA'
import { locales, type Locale } from '@/i18n'

// Main fonts for English and Russian
const inter = Inter({ 
  subsets: ['latin', 'cyrillic'],
  variable: '--font-inter',
  display: 'swap',
  preload: true,
})

const montserrat = Montserrat({ 
  weight: ['400', '500', '600', '700', '800', '900'],
  subsets: ['latin', 'cyrillic'],
  variable: '--font-montserrat',
  display: 'swap',
  preload: true,
})

// Epic fonts for logo and headings
const cinzel = Cinzel({ 
  weight: ['400', '600', '700', '900'],
  subsets: ['latin'],
  variable: '--font-cinzel',
  display: 'swap',
  preload: true,
})

const orbitron = Orbitron({ 
  weight: ['400', '600', '700', '900'],
  subsets: ['latin'],
  variable: '--font-orbitron',
  display: 'swap',
  preload: true,
})

// Georgian font
const notoSansGeorgian = Noto_Sans_Georgian({
  weight: ['400', '500', '600', '700', '800', '900'],
  subsets: ['georgian'],
  variable: '--font-noto-sans-georgian',
  display: 'swap',
  preload: true,
})

export function generateStaticParams() {
  return locales.map((locale) => ({ locale }))
}

export async function generateMetadata({ 
  params
}: { 
  params: Promise<{ locale: Locale }>
}): Promise<Metadata> {
  const { locale } = await params
  const t = await getTranslations({ locale, namespace: 'hero' })
  
  return {
    title: `${t('companyName')} - ${t('subtitle')}`,
    description: t('description', { 
      bless: t('andrew'), 
      soul: t('seed') 
    }),
    keywords: 'full stack developer, web development, React, Next.js, TypeScript, MVP development, production ready, software development, UI/UX design, backend development, frontend development, MU Online, Bless Soul',
    authors: [{ name: 'Andrew Altair' }, { name: 'Seed' }],
    creator: 'Bless & Soul Development Duo',
    publisher: 'Bless & Soul',
    formatDetection: {
      email: false,
      address: false,
      telephone: false,
    },
    metadataBase: new URL(process.env.NEXT_PUBLIC_SITE_URL || 'http://localhost:3000'),
    alternates: {
      canonical: `/${locale}`,
      languages: {
        'en': '/en',
        'ru': '/ru',
        'ua': '/ua',
      },
    },
    openGraph: {
      type: 'website',
      locale: locale === 'en' ? 'en_US' : locale === 'ru' ? 'ru_RU' : 'uk_UA',
      alternateLocale: locale === 'en' ? ['ru_RU', 'uk_UA'] : locale === 'ru' ? ['en_US', 'uk_UA'] : ['en_US', 'ru_RU'],
      url: `/${locale}`,
      title: `${t('companyName')} - ${t('subtitle')}`,
      description: t('description', { bless: t('andrew'), soul: t('seed') }),
      siteName: 'Bless & Soul',
      images: [
        {
          url: '/og-image.png',
          width: 1200,
          height: 630,
          alt: 'Bless & Soul - Development Duo',
        },
      ],
    },
    twitter: {
      card: 'summary_large_image',
      title: `${t('companyName')} - ${t('subtitle')}`,
      description: t('description', { bless: t('andrew'), soul: t('seed') }),
      images: ['/og-image.png'],
      creator: '@blesssoul',
    },
    robots: {
      index: true,
      follow: true,
      googleBot: {
        index: true,
        follow: true,
        'max-video-preview': -1,
        'max-image-preview': 'large',
        'max-snippet': -1,
      },
    },
    icons: {
      icon: [
        { url: '/favicon.svg', type: 'image/svg+xml' },
        { url: '/favicon-16x16.png', sizes: '16x16', type: 'image/png' },
        { url: '/favicon-32x32.png', sizes: '32x32', type: 'image/png' },
      ],
      apple: [
        { url: '/apple-touch-icon.png', sizes: '180x180', type: 'image/png' },
      ],
    },
    manifest: '/manifest.json',
    verification: {
      google: process.env.NEXT_PUBLIC_GOOGLE_VERIFICATION,
      yandex: process.env.NEXT_PUBLIC_YANDEX_VERIFICATION,
    },
  }
}

export default async function LocaleLayout({
  children,
  params
}: {
  children: React.ReactNode
  params: Promise<{ locale: Locale }>
}) {
  const { locale } = await params
  
  // Validate that the incoming locale is valid
  if (!locales.includes(locale)) {
    notFound()
  }

  // Providing all messages to the client side is the easiest way to get started
  const messages = await getMessages({ locale })

  return (
    <html lang={locale} suppressHydrationWarning>
      <head>
        <link rel="preconnect" href="https://fonts.googleapis.com" />
        <link rel="preconnect" href="https://fonts.gstatic.com" crossOrigin="anonymous" />
      </head>
      <body 
        className={`${inter.variable} ${montserrat.variable} ${cinzel.variable} ${orbitron.variable} ${notoSansGeorgian.variable} ${locale === 'ka' ? notoSansGeorgian.className : inter.className}`}
        suppressHydrationWarning
      >
        <NextIntlClientProvider messages={messages}>
          <Providers>
            <ScrollProgress />
            <CustomCursor />
            <EasterEgg />
            <Navbar />
            <main className="min-h-screen">
              {children}
            </main>
            <Footer />
            <ScrollToTop />
            <FloatingCTA />
          </Providers>
        </NextIntlClientProvider>
      </body>
    </html>
  )
}

